package ch.std.jumpstart;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletContext;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.SpringBootTest.WebEnvironment;
import org.springframework.boot.test.web.client.TestRestTemplate;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.http.HttpEntity;
import org.springframework.test.context.ActiveProfiles;

import ch.std.jumpstart.dto.BookDTO;
import ch.std.jumpstart.jpa.Book;

@SpringBootTest(webEnvironment = WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
public class BookControllerTests {

	Logger logger = LoggerFactory.getLogger(BookControllerTests.class);

	@LocalServerPort
	private int port;

	@Autowired
	private ServletContext servletContext;

	@Autowired
	private TestRestTemplate restTemplate;

	private List<BookDTO> bookList = new ArrayList<>();

	@BeforeEach
	public void setup() {
		logger.info("BookControllerTests.setup");
		this.bookList.clear();
		// create test book
		String postUrl = this.getUrl("/rest/book");
		HttpEntity<BookDTO> request = new HttpEntity<>(new BookDTO("1234567890","test", "test", "test"));
		logger.info("BookControllerTests.setup, postUrl = " + postUrl);
		BookDTO testBook = this.restTemplate.postForObject(postUrl, request, BookDTO.class);
		logger.info("BookControllerTests.setup, testBook = " + testBook);
		this.bookList.add(testBook);
	}

	@Test
	public void contextLoads() {
	}

	@Test
	public void testGetBooks() throws Exception {
		String url = this.getUrl("/rest/books?value=test");
		Book[] books = this.restTemplate.getForObject(url, Book[].class);
		assertNotNull(books);
		assertEquals(1, books.length);
		Book book = books[0];
		assertNotNull(book);
		assertEquals("1234567890", book.getIsbn());
	}

	@Test
	public void testGetAllBooks() throws Exception {
		String url = this.getUrl("/rest/books");
		BookDTO[] books = this.restTemplate.getForObject(url, BookDTO[].class);
		assertNotNull(books);
		assertEquals(this.bookList.size(), books.length);
		for (BookDTO book : this.bookList) {
			assertTrue(bookList.contains(book));
		}
	}

	@Test
	public void testGetBookById() throws Exception {
		String url = this.getUrl("/rest/books?value=test");
		BookDTO[] books = this.restTemplate.getForObject(url, BookDTO[].class);
		assertNotNull(books);
		assertEquals(1, books.length);
		BookDTO book = books[0];
		String urlById = this.getUrl("/rest/book/" + book.getId());
		BookDTO bookById = this.restTemplate.getForObject(urlById, BookDTO.class);
		assertEquals(book, bookById);
	}

	public String getUrl(String path) {
		return "http://localhost:" + port + servletContext.getContextPath() + path;
	}
}
