package ch.std.jumpstart.scheduler;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import ch.std.jumpstart.tasks.impl.InitialTask;
import ch.std.jumpstart.tasks.impl.RunnableTask;

@Component
@ConditionalOnProperty(prefix = "jumpstart.scheduler.initial", name = "enabled", havingValue = "true")
public class InitialScheduledTask {

	private static final Logger log = LoggerFactory.getLogger(InitialScheduledTask.class);

	private InitialTaskConfig initialTaskConfig;

	private JdbcTemplate jdbcTemplate;

	public InitialScheduledTask(InitialTaskConfig initialTaskConfig, JdbcTemplate jdbcTemplate) {
		this.initialTaskConfig = initialTaskConfig;
		this.jdbcTemplate = jdbcTemplate;
	}

	@PostConstruct
	public void initialTask() {
		if (!this.initialTaskConfig.isEnabled()) {
			log.info("InitialScheduledTask.initialTask ignore, not enabled");
			// not enabled
			return;
		}
		log.info("InitialScheduledTask.initialTask start");
		InitialTask initialTask = new InitialTask(jdbcTemplate);
		try {
			ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1);
			log.info("InitialScheduledTask.initialTask schedule with delay " + this.initialTaskConfig.delay + "ms");
			scheduledExecutorService.schedule(new RunnableTask(initialTask), this.initialTaskConfig.delay,
					TimeUnit.MILLISECONDS);
		} catch (Exception e) {
			log.error(e.toString(), e);
		}
	}

	@Configuration
	@ConfigurationProperties("jumpstart.scheduler.initial")
	public static class InitialTaskConfig {
		private boolean enabled;
		private long delay;

		public InitialTaskConfig() {
			this.enabled = true;
			this.delay = 0;
		}

		// standard getters and setters
		public boolean isEnabled() {
			return this.enabled;
		}

		public void setEnabled(boolean enabled) {
			this.enabled = enabled;
		}

		public long getDelay() {
			return delay;
		}

		public void setDelay(long delay) {
			this.delay = delay;
		}
	}
}
