package ch.std.jumpstart.repositories;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.util.ArrayList;
import java.util.List;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.domain.Sort;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ch.std.jumpstart.jpa.Author;
import ch.std.jumpstart.jpa.Book;

// see https://www.baeldung.com/spring-boot-testing

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class BookRepositoryJPATest {

	@Autowired
	private BookRepository bookRepository;


	private Book book;
	private List<Book> bookList;

	@Before
	public void setup() {
		book = new Book("978-1617292545", "this is the title", "that's the description", "my publisher");
		this.bookRepository.save(book);
		// insert more books
		this.bookList = new ArrayList<>();
		for (int i = 0; i < 10; i++) {
			Book localBook = null;
			if (i % 2 == 0) {
				localBook = new Book("isbn-" + i, "title-" + i, "description-" + i, "publisher-" + i);
			} else {
				localBook = new Book("isbn-".toUpperCase() + i, "title-".toUpperCase() + i,
						"description-".toUpperCase() + i, "publisher-".toUpperCase() + i);
			}
			this.bookRepository.save(localBook);
			this.bookList.add(localBook);
		}
	}

	@After
	public void tearDown() {
		this.bookRepository.deleteAllInBatch();
	}

	@Test
	public void testFindById() {
		Book foundBook = this.bookRepository.findById(book.getId()).orElse(null);
		assertNotNull(foundBook);
		assertEquals(this.book.toString(), foundBook.toString());
	}

	@Test
	public void testFindByIsbn() {
		Book foundBook = this.bookRepository.findByIsbn("978-1617292545").orElse(null);
		assertNotNull(foundBook);
		assertEquals(this.book.toString(), foundBook.toString());
	}


	@Test
	public void testFindAll() {
		List<Book> foundBookList = this.bookRepository.findAll();
		assertNotNull(foundBookList);
		assertEquals(11, foundBookList.size());
	}

	@Test
	public void testFindAllSorted() {
		List<Book> foundBookList = this.bookRepository.findAll(Sort.by("id").descending());
		assertNotNull(foundBookList);
		assertEquals(11, foundBookList.size());
		assertEquals(this.bookList.get(9).toString(), foundBookList.get(0).toString());
	}

	@Test
	public void testFindByTitle() {
		List<Book> foundBookList = this.bookRepository.findByTitle("title-0");
		assertNotNull(foundBookList);
		assertEquals(1, foundBookList.size());
		assertEquals(this.bookList.get(0).toString(), foundBookList.get(0).toString());
	}

	@Test
	public void testFindByTitleContaining() {
		List<Book> foundBookList = this.bookRepository.findByTitleContaining("itle");
		assertNotNull(foundBookList);
		assertEquals(6, foundBookList.size());
	}

	@Test
	public void testFindByTitleContainingIgnoringCase() {
		List<Book> foundBookList = this.bookRepository.findByTitleContainingIgnoreCase("itle");
		assertNotNull(foundBookList);
		assertEquals(11, foundBookList.size());
	}

	@Test
	public void testFindAllContaining() {
		List<Book> foundBookList = this.bookRepository.findAllContaining("cript");
		assertNotNull(foundBookList);
		assertEquals(6, foundBookList.size());
	}


}
